package examples.testcomposite.aspectj;

/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.pattern.
 *
 * Contributor(s):   
 */
 
import java.io.PrintStream;  
import java.util.Enumeration; 
import ca.ubc.cs.spl.pattern.library.CompositeProtocol;

/**
 * Implements a concrete instance of the composite design pattern.<p> 
 * 
 * It maintains the mapping between composites and their children, defines the
 * component, composite, and leaf roles, and implements facilities to
 * implements methods that work on the whole aggregate structure.
 *
 * <p><i>This is the AspectJ version.</i><p> 
 *
 * Each concrete subaspect does the following things: <UL>
 * <LI> Define which classes are Components and Leafs
 * <LI> (optional) Define methods that operate on the whole aggregate
 *      structure (using visitors)
 * </UL>
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 * @see Component
 * @see CompositeA 
 * @see LeafB
 */

public aspect SampleComposite extends CompositeProtocol {

    /** 
     * Assigns the Composite role to <code>CompositeA</code>
     */
     
    declare parents: Directory implements Composite;   // Role Pattern: Assigning Roles

    /** 
     * Assigns the Leaf role to <code>LeafB</code>
     */

    declare parents: File      implements Leaf;        // Role Pattern: Assigning Roles 
    


    // Test1: Attaching an operation with arguments
    
	/**
	 * helper variable to store recursion depth for pretty printing
	 */

    private static int indent = 0;

    /**
     * Print a number of spaces according to the current recursion depth
     */

    private static void indent() {
		for (int i=0; i<indent; i++)
			System.out.print(" ");
	} 
	
/*
	public int sizeOnDisk(Component c) {
	    return c.sizeOnDisk();
	}
	
	private int Component.sizeOnDisk() {
	    Enumeration e = SampleComposite.aspectOf().recurseFunction(this, new FunctionVisitor() {
	        public Object doFunction(Component c) {
	            return new Integer(c.sizeOnDisk());
	        }
	    });
	    
	    int sum =0;
	    while (e.hasMoreElements()) {
	        sum += ((Integer)e.nextElement()).intValue();
	    }
	    return sum;
	
    }

    private int File.sizeOnDisk() {return id;}
*/

    public int sizeOnDisk(Component c) {     // Does not work
        return c.sizeOnDisk();
    }

    private int Component.sizeOnDisk() {return 0;}
    
    private int Directory.sizeOnDisk() { 
        int sum = 0; 
        java.util.Enumeration children;
        for (children = SampleComposite.aspectOf().getAllChildren(this); children.hasMoreElements(); ) {
            sum += ((Component)children.nextElement()).sizeOnDisk();
        }
        return sum;
    }
    
    private int File.sizeOnDisk()      { return id; }


/*    public int sizeOnDisk(Component c) {        // Works
        int sum = 0;
        if (c instanceof File) {
            sum = ((File)c).id;
        } else if (c instanceof Directory) {
            java.util.Enumeration children;
            for (children = SampleComposite.aspectOf().getAllChildren(c); children.hasMoreElements(); ) {
                sum += sizeOnDisk((Component)children.nextElement());
            } 
        }
        return sum;
    }
*/

    /**
     * Provides a client-accessible method that pretty-prints the 
     * structure of the aggregate structure using a Visitor
     *
     * @param s the PrintStram to print to
     */

    public void Component.printStructure(PrintStream s) {
        indent();
        s.println("<Component>"+this);
    } 
    
    /**
     * Implements <code>printStructure</code> for Composites: The indent 
     * is appropriately updated and the method call is forwarded to all 
     * children.
     *
     * @param s the PrintStram to print to
     */

    public void Composite.printStructure(final PrintStream s) {
        indent();
        s.println("<Composite>"+this);
        indent +=4;
        SampleComposite.aspectOf().recurseOperation(this, new Visitor() { public void doOperation(Component c) { c.printStructure(s); } } );
        indent -=4;
    }

    /**
     * Implements <code>printStructure</code> for Leafs.
     *
     * @param s the PrintStram to print to
     */

    public void Leaf.printStructure(PrintStream s) {
        indent();
        s.println("<Leaf>"+this);
    }
    
 
 
 
    
    // Test2: Collecting statistics on the structure (aggregation)
    
    /**
     * Provides a client-accessible method that pretty-prints the 
     * structure of the aggregate structure using a FunctionVisitor. 
     * Calculates the sum of all Leaf IDs in the structure 
     *
     * @returns the sum of leaf ids of all elements in this structure
     */ 
     
    public int Component.subSum() {
        return 0;
    }

    /**
     * Implements <code>subSum()</code> for Composites: The method call 
     * is forwarded to all children, then the results are summed up.
     *
     * @returns the sum of leaf ids of all elements in this structure
     */

    public int Composite.subSum() {                     // Collects the sum of all Leaves in the structure
        Enumeration enum = SampleComposite.aspectOf().recurseFunction(this, new FunctionVisitor() { 
            public Object doFunction(Component c) { 
                return new Integer(c.subSum()); 
            }
        }); 
        
        int sum = 0;
        while (enum.hasMoreElements()) {
            sum += ((Integer) enum.nextElement()).intValue();
            System.out.println("Sum = "+sum);
        }
        return sum;
    }
    
    /**
     * Implements <code>subSum()</code> for Leafs: Simply returns
     * the Leaf's ID.
     *
     * @returns the leaf id
     */

    public int File.subSum() {
        return id;
    }
}       

