package examples.testcomposite.aspectj;


/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.pattern.
 *
 * Contributor(s):   
 */
 
import java.util.Enumeration;

/**
 * Implements the driver for the composite design pattern example.<p> 
 *
 * Intent: <i>Compose objects into tree structures torepresent part-whole 
 * hierarchies. Composite lets clients treat individual objects and 
 * compositions of objects uniformly.</i><p>
 *
 * Participatng classes are <code>Directory</code>s as <i>Composite</i>s,
 * and a <code>File</code> as <i>Leaf</i>. Both implement the 
 * <i>Component</i> interface.<p>
 *
 * This example creates a simple structure as follows: Composite c1 has 
 * three children: l1, c2, and l3. c2 has l2 as a child.
 * Compact notation: c1(l1, c2(l2), l3)
 *
 * <p><i>This is the AspectJ version.</i><p> 
 *
 * Composites and Leafs do not need to know about their role in the pattern.
 * This example also illustrates how to define methods that collect 
 * information from the whole aggreagate structure (using visitors).
 * One of them prints the compiste structure, the other one collects the
 * sum of the values of all leaves in the structure.
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 * @see Component
 * @see Directory 
 * @see File
 */
  
  public class Main {  
    
	/**
	 * helper variable to store recursion depth for pretty printing
	 */
	 
    static int indent = 0;  
    
    /**
     * Print a number of spaces according to the current recursion depth
     */
     	
    private static void indent() {
        for (int i=0; i<indent; i++)
            System.out.print(" ");
    }
    
    /** 
     * Pretty-prints a recursive composite structure 
     *
     * @param comp the component denoting the entry point into the structure
     */
     
    private static void printStructure(SampleComposite.Component comp) {   
        indent();
        System.out.println("Showing: "+comp);
        indent +=4;                
        for (Enumeration enum = SampleComposite.aspectOf().getAllChildren(comp); enum.hasMoreElements();) {
            printStructure((SampleComposite.Component) enum.nextElement());
        }
        indent -= 4;
    }


    /**
     * This example creates a simple structure as follows: Composite c1 has 
     * three children: l1, c2, and l3. c2 has l2 as a child.
     * Compact notation: c1(l1, c2(l2), l3) <p> 
     *
     * Also, this example illustrates how to define methods that collect 
     * information from the whole aggreagate structure (using visitors).
     * One of them prints the compiste structure, the other one collects the
     * sum of the values of all leaves in the structure.   
     */

    public static void main(String[] args) { 
        
        System.out.println("\n<<< Testing implementation AOP3 of Composite pattern >>>\n");
        System.out.print  ("Creating Composite structure ...");

        Directory composite1 = new Directory(1);
        Directory composite2 = new Directory(2);

        File      leaf1      = new File(1024);
        File      leaf2      = new File(128);
        File      leaf3      = new File(4096); 
        
        SampleComposite.aspectOf().addChild(composite1, leaf1);
        SampleComposite.aspectOf().addChild(composite1, composite2);
        SampleComposite.aspectOf().addChild(composite2, leaf2);
        SampleComposite.aspectOf().addChild(composite1, leaf3);
        
        System.out.println("done."); 
        System.out.println("This is the Structure:");
        
        printStructure(composite1);
        
        System.out.println("\nCalling printStructure(PrintStream) on Composition.Components ...");
        composite1.printStructure(System.out);  
        System.out.println("... done.");

        System.out.println("\nCalling subSum():int on the structure ...");
        System.out.println("The leaf id sum is: "+SampleComposite.aspectOf().sizeOnDisk(composite1));  
        System.out.println("... done.");

        System.out.println("\n<<< Test completed >>>\n");
    }
}