package examples.interpreter.aspectj;

/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.patterns.
 *
 * Contributor(s):   
 */

/**
 * Attaches implementations of <code>replace(String, BooleanExp)<code> and
 * <code>copy()</code> methods to all concrete <code>BooleanExp</code>s. 
 *
 * The very nature of the interpreter pattern introduces coupling between all
 * participants. Unfortunately, removeing the pattern code from the 
 * participants does not work as nicely here as with other patterns. The
 * reason is that the roles are defining, i.e., each participant's 
 * functionality is determined (only) by its role. Removing pattern
 * specific code into an aspect would leave the <i>Expressions</i> etc.
 * empty and would make the aspect a monolithic module. <p>
 *
 * However, it is still possible to augment or change the behaviour of the
 * system without changing all participant classes. To show this, we 
 * assumed that <code>BooleanExp.replace(String, BooleanExp)</code> and
 * <code>BooleanExp.copy()</code> were added later. The code for these
 * methods is placed in the aspect, so that other classes did not have to
 * change (we only changed the interface, but even that was not necessary).<p>
 *
 * In general, however, this pattern does not lend itself nicely to 
 * aspectification.<p>
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 * @see BooleanExp
 */

public aspect BooleanInterpretation {
                                            
// AndExp

    /**
     * Replaces a variable with an expression. Has no effect.
     *
     * @param name the name of the variable
     * @param exp the expression to replace the variable
     * @returns a copy of this expression with the variable replaced
     */

	public BooleanExp AndExp.replace(String name, BooleanExp exp) {
		return new AndExp(op1.replace(name, exp), op2.replace(name,exp));
	}
		
    /**
     * Copies this expression
     *
     * @returns the copied expression
     */
     
	public BooleanExp AndExp.copy() {
		return new AndExp(op1.copy(), op2.copy());
	}                                            

// BooleanConstant

    /**
     * Replaces a variable with an expression. Has no effect.
     *
     * @param name the name of the variable
     * @param exp the expression to replace the variable
     * @returns a copy of this expression with the variable replaced
     */

	public BooleanExp BooleanConstant.replace(String name, BooleanExp exp) {
		return exp;   // ???
	}
	
    /**
     * Copies this expression
     *
     * @returns the copied expression
     */
     
	public BooleanExp BooleanConstant.copy() {
		return new BooleanConstant(value);
	}

// OrExp
	
    /**
     * Replaces a variable with an expression. Has no effect.
     *
     * @param name the name of the variable
     * @param exp the expression to replace the variable
     * @returns a copy of this expression with the variable replaced
     */

	public BooleanExp OrExp.replace(String name, BooleanExp exp) {
		return new OrExp(op1.replace(name, exp), op2.replace(name,exp));
	}
		
    /**
     * Copies this expression
     *
     * @returns the copied expression
     */
     
	public BooleanExp OrExp.copy() {
		return new OrExp(op1.copy(), op2.copy());
	}
	
// VariableExp

    /**
     * Replaces a variable with an expression. Has no effect.
     *
     * @param name the name of the variable
     * @param exp the expression to replace the variable
     * @returns a copy of this expression with the variable replaced
     */

	public BooleanExp VariableExp.replace(String name, BooleanExp exp) {
		if (name.equals(this.name)) {
			return exp.copy();
		} else {
			return new VariableExp(this.name);
		}
	}
	
    /**
     * Copies this expression
     *
     * @returns the copied expression
     */
     
	public BooleanExp VariableExp.copy() {
		return new VariableExp(name);
	}

// NotExp

	
    /**
     * Replaces a variable with an expression. Has no effect.
     *
     * @param name the name of the variable
     * @param exp the expression to replace the variable
     * @returns a copy of this expression with the variable replaced
     */

	public BooleanExp NotExp.replace(String name, BooleanExp exp) {
		return new NotExp(this.exp.replace(name, exp));
	}
		
    /**
     * Copies this expression
     *
     * @returns the copied expression
     */
     
	public BooleanExp NotExp.copy() {
		return new NotExp(exp.copy());
	}		
}

