package examples.chainOfResponsibility.java;

/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.patterns.
 *
 * Contributor(s):   
 */
 

/**
 * Implements a GUI-motivated example for the Chain Of Rspsonsibility design
 * pattern.<p> 
 *
 * Intent: <i>Avoid coupling the sender of a request to its receiver by giving
 * more than one object a chance to handle the request. Chain the receiving 
 * objects and pass the request along the chain until an object handles it.
 * </i><p>
 *
 * Participatng objects are a <code>Frame</code>, a <code>Panel</code>, and 
 * <code>Button</code>
 *
 * A click on the button triggers an event (request) that gets passed along
 * the widget hierarchy (button -> panel -> frame).
 *
 * The <code>Handler</code> interface defines the <code>handleRequest()</code>
 * method for asking an object if it is willing to handle the request.  
 *
 * <p><i>This is the Java version.</i><p>    
 *
 * In this version, the event not handled by button and panel. Only frame 
 * handles the event.
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 * @see Button
 * @see Panel
 * @see Frame
 * @see Handler
 */

public class Main {
	
    /**
     * Implements the driver for the chain of responisbility example. 
     * It creates a simple GUI consisting of a <code>Button</code> in a 
     * <code>Panel</code> in a <code>Frame</code>. 
     *
     * Clicking the button will start a request, that gets passed on
     * along the following chain: button, panel, frame. the frame handles 
     * the event. <p>
     *
     * The following messages should be observed on a buttonclick: <OL>
     * <LI> An Indication that the request was started
     * <LI> Request received by: Button (unhandled: forwarded)
     * <LI> Request received by: Panel (unhandled: forwarded)
     * <LI> Request received by: Frame (handled)
     * </UL>
     */ 

	public static void main(String[] args) {  
	    
		Frame  frame  = new Frame("Chain of Responsibility");
		Panel  panel  = new Panel(frame);
		Button button = new Button("Click me to see the Chain Of Responsibility pattern in action!", panel);
		
		frame.getContentPane().add(panel);
		panel.add(button);
		
		frame.pack();
		frame.setVisible(true);
	}
}